/*----------------------------------------------------------------------------

NAME

	DuoHand.h.  Header file demonstrating one way to implement two-handed
	input with a Wacom Intuos tablet.  This example uses the CSR_MASK 
	(cursor mask) Wintab extension which is described in Appendix B of the 
	Wintab specification.

COPYRIGHT

	Copyright (C) 1998  LCS/Telegraphics
	Copyright (c) Wacom Company, Ltd. 2014 All Rights Reserved
	All rights reserved.

	The text and information contained in this file may be freely used,
	copied, or distributed without compensation or licensing restrictions.

-----------------------------------------------------------------------------*/

#ifndef DUOHAND_H
#define DUOHAND_H

#define WACOM_DEBUG

// converts FIX32 to double
#define	FIX_DOUBLE(x)	((double)(INT(x))+((double)FRAC(x)/65536))
#define	pi		3.14159265359

//////////////////////////////////////////////////////////////////////////////

#define PROGRAM_NAME			"DuoHand"
#define MAIN_WNDCLASS_NAME "CursorMaskTestWClass"

#define EXTENSION_NOT_FOUND ( ( LONG ) ( -1 ) )

//////////////////////////////////////////////////////////////////////////////

// Number of bytes in cursor mask.  Note that Wacom tablets only use the 
// first 6 bits of this mask.  These 6 bits correspond to the 
// EWINTAB_CURSOR_INDEXes 0-5 which are described below.
#define SIZE_CURSOR_MASK ( ( UINT ) 16 )

// Wacom tablets currently support sending data for two cursors which are
// simultaneously in proximity of the tablet.  
#define NUM_SIMULTANEOUS_CURSORS_SUPPORTED ( ( UINT ) 2 )

// We are going to open two Wintab contexts because we are supporting
// two simultaneous cursors.
#define NUMBER_WINTAB_CONTEXTS NUM_SIMULTANEOUS_CURSORS_SUPPORTED

// The context at the top of the Wintab z-order of Wintab contexts will handle
// the 'Dominant' cursor.  The dominant cursor is the cursor which moves the
// system pointer.
#define DOMINANT_HAND_CONTEXT_INDEX ( ( UINT ) 1 )

// Immediately below the top context is the 'Nondominant' context.  This 
// context detects when a new cursor enters proximity of the tablet and
// also handles the cursor which does not move the system cursor.
#define NONDOMINANT_HAND_CONTEXT_INDEX ( ( UINT ) 0 )

// When a Wintab context receives a packet, Wintab puts the packet in a queue
// which is unique for each context.  The default size of the queue is eight
// packets.  The queue will overflow very quickly because the tablet will send
// data at up to 200 packets a second.  When the queue overflows and Wintab
// discards subsequent packets until there is room in the queue.  When the 
// queue overflows, you also you stop receiving WT_PACKET messages.  300
// packets should guarantee the queue doesn't overflow.
	
#define WINTAB_CTX_QUEUE_SIZE 300

// Wintab cursors.  Note that Wintab considers inverted and non-inverted
// stylii to be different cursors.  If multiple tablets were supported,
// these cursors would begin at the value returned by DVC_FIRSTCSR.

#define EWINTAB_CURSOR_INDEX_PUCK_0					0
#define EWINTAB_CURSOR_INDEX_STYLUS_0				1
#define EWINTAB_CURSOR_INDEX_STYLUS_INVERTED_0	2
#define EWINTAB_CURSOR_INDEX_PUCK_1					3
#define EWINTAB_CURSOR_INDEX_STYLUS_1				4
#define EWINTAB_CURSOR_INDEX_STYLUS_INVERTED_1	5

#define MAX_EWINTAB_CURSOR_INDEX		EWINTAB_CURSOR_INDEX_STYLUS_INVERTED_1

#define EWINTAB_CURSOR_INDEX_NONE ( ( UINT ) -1 )

#define CSR_MASK_NO_CURSORS ( ( BYTE ) 0 )

#define CSR_MASK_ALL_CURSORS									\
	( ( 1 << EWINTAB_CURSOR_INDEX_PUCK_0  ) |				\
	( 1 << EWINTAB_CURSOR_INDEX_STYLUS_0 ) |				\
	( 1 << EWINTAB_CURSOR_INDEX_STYLUS_INVERTED_0 ) |	\
	( 1 << EWINTAB_CURSOR_INDEX_PUCK_1 ) |					\
	( 1 << EWINTAB_CURSOR_INDEX_STYLUS_1 )	|				\
	( 1 << EWINTAB_CURSOR_INDEX_STYLUS_INVERTED_1 ) )

// The upper two bits of the WTInfo value returned for 
// WTInfo( WTI_CURSORS + wCursorIndex, ...)identify general cursors types as
// shown here.

#define CSR_TYPE_GENERAL_MASK			( ( UINT ) 0xC000 )
#define CSR_TYPE_GENERAL_PENTIP		( ( UINT ) 0x4000 )
#define CSR_TYPE_GENERAL_PUCK			( ( UINT ) 0x8000 )
#define CSR_TYPE_GENERAL_PENERASER	( ( UINT ) 0xC000 )

// Pen tip or eraser
#define CSR_TYPE_GENERAL_PEN			( ( UINT ) 0x4000 )

// This mask can be used to more specifically identify a cursor using the
// CSR_TYPE value.

#define CSR_TYPE_SPECIFIC_MASK ( ( UINT ) 0x0F06 )

#define CSR_TYPE_SPECIFIC_STYLUS_BITS			( ( UINT ) 0x0802 )
#define CSR_TYPE_SPECIFIC_AIRBRUSH_BITS		( ( UINT ) 0x0902 )
#define CSR_TYPE_SPECIFIC_4DMOUSE_BITS			( ( UINT ) 0x0004 )
#define CSR_TYPE_SPECIFIC_LENSCURSOR_BITS		( ( UINT ) 0x0006 )

//////////////////////////////////////////////////////////////////////////////
// The CSR_TYPE WTInfo data item is new to Wintab 1.2 and is not defined
// in the Wintab 1.26 SDK, so we have to define it.
#ifndef CSR_TYPE
#	define CSR_TYPE 20
#endif

//////////////////////////////////////////////////////////////////////////////
// This structure contains both information about the context and the cursor
// which the context is handling.
//
struct tagWintabContextInfo
{
// Wintab context handle obtained through WTOpen()
	HCTX hCtx;

// A bitfield describing which Wintab cursors are handled by this context.  
	BYTE cCursorMask[ SIZE_CURSOR_MASK ];

// The general type of the Wintab cursor, obtained from 
//	WTInfo( WTI_CURSORS + wCursorIndex, CSR_TYPE, ... ). This data item is new
// to Wintab version 1.2.
	UINT wCursorType;

// Physical ID of the Wintab cursor handled by this context.  Obtained from
// WTInfo( WTI_CURSORS + wCursorIndex, CSR_PHYSID, ... ).
	UINT wCursorPhysicalID;			

// CSR_TYPE and CSR_PHYSID together uniquely identify a cursor.

// The rest the structure contains packet data from the last packet read
// for the cursor handled by the context.

// Wintab cursor index (EWINTAB_CURSOR_INDEX_PUCK_0, etc.).  
// EWINTAB_CURSOR_INDEX_NONE means no cursor is being handled (no cursor is 
// in proximity in the context.)
	UINT eWintabCursor;

// Tablet coordintates
	UINT wCurX;
	UINT wCurY;
	UINT wCurZ;
	
// Tilt and rotation information
	ORIENTATION	curOrientation;

// Tip pressure
	UINT wCurPressure;

// Non-tip button pressure.
	UINT wCurTangentPressure;

// Pen with which to display data for this packet
	HPEN hPen;

// Which context are we?
	BOOL bIsDominant;
};

typedef struct tagWintabContextInfo WINTAB_CONTEXT_INFO;

//////////////////////////////////////////////////////////////////////////////

#define DOMINANT_COLOR			RGB( 0, 255, 0 )
#define NONDOMINANT_COLOR		RGB( 0, 0, 255 )

//////////////////////////////////////////////////////////////////////////////
int PASCAL WinMain(HINSTANCE, HINSTANCE, LPSTR, int);
BOOL InitApplication(HANDLE);
BOOL InitInstance(HANDLE, int);

void Cleanup( void );

LRESULT FAR PASCAL MainWndProc(HWND, unsigned, WPARAM, LPARAM);
BOOL FAR PASCAL DlgAboutProc(HWND, unsigned, WPARAM, LPARAM);
UINT SavePacket( HCTX hCtx, UINT wPacketNumber );
UINT TabletInit( HWND hWnd );
BOOL HandleWTCursorChange( HWND hWnd, WPARAM wPacketNumber, LPARAM hCtxIn );
BOOL HandleWTProximity( HWND hWnd, WPARAM hCtxIn, 
	LPARAM dwTypeOfProximityChange );
BOOL IsCursorMaskExtensionAvailable( void );

UINT ShowWintabCursor( HWND hWnd, HDC hDC, WINTAB_CONTEXT_INFO *pWTContextInfo );
void ShowCursorDescriptions( HWND hWnd, HDC hDC );

BOOL SetCursorMask( HWND hWnd, DWORD wParam );

BOOL IsDominantCursor( UINT wCursor );
BOOL IsNonDominantCursor( UINT wCursor );
BOOL IsPen( UINT wCursor );

char *GetGeneralCursorName( UINT wCursorIndex );
char *GetSpecificCursorName( UINT wCursorType );

//////////////////////////////////////////////////////////////////////////////

#endif // DUOHAND_H




