/*----------------------------------------------------------------------------

	NAME
		DuoHand.c

	PURPOSE

		Demonstrates one method to implement dual track (two handed) support
		for Wacom Intuos2 tablets using the Wintab CSR_MASK extension.

		This example creates two contexts.  One context handles the dominant
		hand cursor and the other context is for the non-dominant hand cursor.
 
		The rules for which cursor is dominant and which non-dominant are as
		as follows:
			if new cursor is the only cursor on the tablet
				make the new cursor move the system cursor
			else
				if puck is already on the tablet
					if new cursor is stylus
						make stylus move system cursor
					endif
				endif
			endif 

		This example demonstrates handling CSR_MASK for a single tablet only.

		TERMS

			Dominant cursor - The tablet device (pen, puck, airbrush, etc.) which
				should always move the system (screen) pointer.

			Nondominant cursor - The tablet device which which only moves the
				screen pointer if a dominant tablet cursor is not in proximity of 
				the tablet.

			Dominant context - The Wintab context which moves the system pointer.
				The dominant context is on the top of the Wintab context overlap
				order when DuoHand.exe is the foreground application.  Wintab
				moves the system pointer in when data is tablet data received in 
				this context.

			Nondominant context.- The Wintab context which does not move the
				system pointer.  The nondominant context lies underneath the
				dominant context in the Wintab context overlap order.  Our program
				must paint a pointer on the screen for this cursor.

  COPYRIGHT
		Copyright (C) 1998  LCS/Telegraphics
		Copyright (c) Wacom Company, Ltd. 2014 All Rights Reserved
		All rights reserved.

		The text and information contained in this file may be freely used,
		copied, or distributed without compensation or licensing restrictions.

----------------------------------------------------------------------------*/

#include	<windows.h>
#include	<stdio.h>
#include	<assert.h>
#include	<stdarg.h>

#include	"wintab.h"		// NOTE: get from wactab header package
#include	"Utils.h"

// What Wintab packet data items we want.  PK_CURSOR identifies which cursor 
// (0-5) generated the packet.  PACKETDATA must be defined before including
// pktdef.h

#define	PACKETDATA	( PK_X | PK_Y | PK_Z | PK_CURSOR | PK_BUTTONS | \
	PK_NORMAL_PRESSURE | PK_ORIENTATION )

// No data items are relative data items, all are absolute data items.
#define	PACKETMODE	0

#include	"pktdef.h"		// NOTE: get from wactab header package
#include	<math.h>
#include	<string.h>
#include	<memory.h>

#include	"DuoHand.h"
#include	"resource.h"

HANDLE ghProgramInstance = NULL;

char*	gpszProgramName = "Tablet Controls Sample";

double prsFactor = 1;			/* Pressure factor */
double rotFactor = 1;			/* Rotation factor */
double aziFactor = 1;			/* Azimuth factor */
double altFactor = 1;			/* Altitude factor */
double altAdjust = 1;			/* Altitude zero adjust */

BOOL tilt_support = TRUE;	/* Is tilt supported */
BOOL rot_support = FALSE; /* Is rotation supported */
    
WINTAB_CONTEXT_INFO gWTContextInfo[ NUMBER_WINTAB_CONTEXTS ];

// Ignore warnings about using unsafe string functions.
#pragma warning( disable : 4996 )

//////////////////////////////////////////////////////////////////////////////

int PASCAL WinMain( 
	HINSTANCE hInstance, 
	HINSTANCE hPrevInstance, 
	LPSTR lpCmdLine,
	int nCmdShow )
{
	MSG msg;

	WACOM_TRACE( "WinMain()\n" );

	if ( !hPrevInstance )
	{
		if ( !InitApplication( hInstance ) )
		{
			return FALSE;
		}
	}

	if ( !InitInstance(hInstance, nCmdShow ) )
	{
		return FALSE;
	}

	while ( GetMessage( &msg,NULL,0,0 ) ) 
	{
		TranslateMessage( &msg );
		DispatchMessage( &msg );
	}

	Cleanup( );

	return msg.wParam;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Registers window class used by the app
// Returns
//		TRUE on success
//		FALSE on error
//
BOOL InitApplication( HANDLE hInstance )
{
	WNDCLASS wc;

	char szClassWndName[ 100 ];

	WACOM_TRACE( "InitApplication()\n" );

	strcpy( szClassWndName, MAIN_WNDCLASS_NAME );

	wc.style = 0;
	wc.lpfnWndProc = MainWndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = hInstance;
	wc.hIcon = LoadIcon( NULL, IDI_APPLICATION );
	wc.hCursor = LoadCursor( NULL, IDC_ARROW );
	wc.hbrBackground = ( HBRUSH ) ( COLOR_APPWORKSPACE + 1 );
	wc.lpszMenuName =  MAKEINTRESOURCE( IDM_MAIN );
	wc.lpszClassName = szClassWndName;

	return RegisterClass( &wc ) ? TRUE : FALSE ;
}


//////////////////////////////////////////////////////////////////////////////

BOOL InitInstance( HANDLE hInstance, int nCmdShow )
{
	HWND			hWnd;

	char			WName[50];			// String to hold window name
	struct		tagAXIS TpOri[3];	// The capabilities of tilt
	struct		tagAXIS TpPrs;		// The capabilities of pressure
	double		tpvar;				// A temp for converting fix to double

	UINT wWTInfoRetVal;

	WACOM_TRACE( "InitInstance()\n" );

	WACOM_ASSERT( hInstance );

	ghProgramInstance = hInstance;

	memset( &gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ], 0, 
		sizeof( WINTAB_CONTEXT_INFO ) );

	memset( &gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ], 0, 
		sizeof( WINTAB_CONTEXT_INFO ) );

	gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hPen =
		CreatePen( PS_SOLID,	1, DOMINANT_COLOR );

	gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hPen =
		CreatePen( PS_SOLID,	1, NONDOMINANT_COLOR );

	WACOM_ASSERT( gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hPen );
	if ( !gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hPen )
	{
		return FALSE;
	}

	WACOM_ASSERT( gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hPen );
	if ( !gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hPen )
	{
		return FALSE;
	}

	if ( !LoadWintab( ) )
	{
		ShowError( "Wintab not available" );
		return FALSE;
	}

	WACOM_ASSERT( gpWTInfoA );

// check if it's a WACOM tablet
	wWTInfoRetVal = gpWTInfoA( WTI_DEVICES, DVC_NAME, WName );
	if ( wWTInfoRetVal < 5 )
	{
		ShowError( "No tablet attached" );
		return FALSE;
	}

// Actually, some non-Wacom tablets will return 'Wacom' so older applications
// will support eraser and pressure.
	if ( strncmp( WName, "WACOM", 5 ) ) 
	{
		ShowError( "Attached tablet is not a Wacom tablet." );
		return FALSE;
	}

	if ( !IsCursorMaskExtensionAvailable( ) )
	{
		ShowError( "This Wintab doesn't support cursor mask extension." );
		return FALSE;
	}

	if ( gpWTInfoA( WTI_DEVICES, DVC_NPRESSURE, &TpPrs ) ) 
	{
	// make 100 the maximum pressure
		prsFactor = ( double )( TpPrs.axMax + 1 ) / 128;
	}
	else
	{
		WACOM_ASSERT( 0 );
		return FALSE;
	}

// get info about tilt and rotation
	tilt_support = gpWTInfoA( WTI_DEVICES, DVC_ORIENTATION, &TpOri );
	if ( !tilt_support )
	{
		ShowError("Tilt not support. Not an Intuos tablet." );
		return FALSE;
	}

	if ( tilt_support )
	{
	// check rotation
		if ( TpOri[2].axResolution ) 
		{
		// convert rotation resulution to double
			tpvar = FIX_DOUBLE( TpOri[2].axResolution );

		// convert from resolution to radians
			rotFactor = tpvar / ( 2 * pi );  

			rot_support = TRUE;
		}

	// does the tablet support azimuth and altitude
		if ( TpOri[0].axResolution && TpOri[1].axResolution ) 
		{
		// convert azimuth resulution to double
			tpvar = FIX_DOUBLE( TpOri[0].axResolution );

		// convert from resolution to radians
			aziFactor = tpvar/( 2 * pi );  
			
		// convert altitude resolution to double
			tpvar = FIX_DOUBLE( TpOri[1].axResolution );

		// scale to arbitrary value to get decent line length
			altFactor = tpvar / 500; 

		 // adjust for maximum value at vertical
			altAdjust = ( double ) TpOri[1].axMax / altFactor;
		}
		else 
		{	
		// no, so dont do tilt stuff 
			tilt_support = FALSE;
		}
	}
	
// Create a main window for this application instance.
	wsprintf( WName, "CursorMask: %x", ghProgramInstance );
	hWnd = CreateWindow(
		MAIN_WNDCLASS_NAME,
		WName,
		WS_OVERLAPPEDWINDOW,
		0,
		0,
		CW_USEDEFAULT,
		CW_USEDEFAULT,
		NULL,
		NULL,
		ghProgramInstance,
		NULL
	);

	WACOM_ASSERT( hWnd );
	if ( !hWnd )
	{
		return FALSE;
	}

	ShowWindow(hWnd, nCmdShow);
	UpdateWindow(hWnd);

	return TRUE;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Checks to see if cursor mask extension is supported by the Wintab
//		on this machine.
// Returns
//		TRUE if cursor mask is supported
//		FALSE if not supported.
//
BOOL IsCursorMaskExtensionAvailable( void )
{
	UINT wWTInfoRetVal;
	UINT wExtensionTag;
	UINT wNumExtensions;
	UINT wCurExtensionIndex;

	UINT wMajorVersion;
	UINT wMinorVersion;
	UINT wSpecVersion;

	WACOM_TRACE( "IsCursorMaskExtensionAvailable()\n" );

	WACOM_ASSERT( gpWTInfoA );

// how many Wintab extensions are supported?

	wWTInfoRetVal = gpWTInfoA( WTI_INTERFACE, IFC_NEXTENSIONS, &wNumExtensions );
	WACOM_ASSERT( wWTInfoRetVal == sizeof( UINT ) );

	WACOM_ASSERT( wNumExtensions );
	if ( !wNumExtensions )
	{
		return FALSE;
	}

// Find which extension index corresponds to the cursor mask extension

	for(	wCurExtensionIndex = 0, wExtensionTag = !WTX_CSRMASK,
				wWTInfoRetVal = -1;
			wWTInfoRetVal && ( wExtensionTag != WTX_CSRMASK ) ;
			wCurExtensionIndex++ )
	{
		wWTInfoRetVal = gpWTInfoA( WTI_EXTENSIONS + wCurExtensionIndex, 
			EXT_TAG, &wExtensionTag );
	}

	WACOM_ASSERT( wExtensionTag == WTX_CSRMASK );
	if ( wExtensionTag != WTX_CSRMASK )
	{
		return FALSE;
	}

// Does this Wintab version supports Wacom cursor id information 
// (version 1.2)?  This version has CSR_TYPE which together with CSR_PHYSID 
// uniquely identifies a Wacom cursor.
// 
// As described at the developer support page at www.wacomeng.com, CSR_TYPE
// is a bit field of which identify the general type of Intuos transducer.
// For example, CSR_TYPE currently contains information which identifies
// the following cursors:
//		General stylus:	(CSR_TYPE & 0x0F06) == 0x0802
//		Airbrush:			(CSR_TYPE & 0x0F06) == 0x0902
//		Spuck:				(CSR_TYPE & 0x0F06) == 0x0004 
//		5 button puck:		(CSR_TYPE & 0x0F06) == 0x0006
// 
// NOTE: The currently shipping Wintab SDK (1.26) is for version Wintab 1.1 
// and does not include the the symbolic constant CSR_TYPE.  So, if you want
// to add Intuos airbrush and spuck support, you will have to define
// CSR_TYPE yourself.  CSR_TYPE is defined as 20.

	wWTInfoRetVal = gpWTInfoA( WTI_INTERFACE, IFC_SPECVERSION, &wSpecVersion );
	WACOM_ASSERT( wWTInfoRetVal == sizeof( WORD) ) ;

	wMajorVersion = HIBYTE( LOWORD( wSpecVersion ) );
	wMinorVersion = LOBYTE( LOWORD( wSpecVersion ) );

	if ( ( wMajorVersion < 1 ) && ( wMinorVersion < 2 ) )
	{
		WACOM_ASSERT( 0 );
		return FALSE;
	}		

	return TRUE;
}


//////////////////////////////////////////////////////////////////////////////

UINT TabletInit( HWND hWnd )
{
	LOGCONTEXT logContext;

	AXIS TabletX;
	AXIS TabletY;

	BOOL bResult;

	UINT wCurContext;
	UINT wWTInfoRetVal;

	WACOM_TRACE( "TabletInit()\n" );

	WACOM_ASSERT( hWnd );
	WACOM_ASSERT( IsWindow( hWnd ) );
	WACOM_ASSERT( ghProgramInstance );
	WACOM_ASSERT( gpWTInfoA );
	WACOM_ASSERT( gpWTOpenA );
	WACOM_ASSERT( gpWTQueueSizeSet );

// Ask Wintab to fill in a logical context struture with values
// which constitute a 'system' context.  We will use these values
// as a kind of template with which to open our own contexts with
// WTOpen().
// 
// When you open your own context with values filled-in by WTI_DEFSYSTCTSX,
// your context will return x and y coordinates in screen coordinates with
// the Intuos Wintab and the CXO_SYSTEM bit of the lcOptions field will be set 
//	(unless you clear it before you open it).  If CXO_SYSTEM is set, Wintab 
//	moves the screen (mouse) cursor when data is received in your context.  
	
//	Some Wintab example programs open a 'digitizer', which means they
// call WTInfo( WTI_DEFCONTEXT) to get the values with which to open
// their own context.  If you use the WTI_DEFCONTEXT values without
// modifying them the CXO_SYSTEM will be clear and the mouse cursor
// will NOT move when your context receives data.  You will also
// get x and y coordinates in tablet coordinates, not screen coordinates.

	wWTInfoRetVal = gpWTInfoA( WTI_DEFSYSCTX, 0, &logContext );
	WACOM_ASSERT( wWTInfoRetVal );

// We process WT_PACKET (CXO_MESSAGES) messages in this example, 
// but you could just call Wintab packet functions in a loop on mouse 
// button downs to get the tablet data.  Since CXO_CSRMESSAGES
// is set, we will also receive WT_CSRCHANGE messages, which tell
// us when a new cursor enters our context.

	logContext.lcOptions |= ( CXO_MESSAGES | CXO_CSRMESSAGES );

// What data items we want to be included in the tablet packets
	logContext.lcPktData = PACKETDATA;

// Which packet items should show change in value since the last
// packet (referred to as 'relative' data) and which items
// should be 'absolute'.
	logContext.lcPktMode = PACKETMODE;

// This bitfield determines whether or not this context will receive
// a packet when a value for each packet field changes.  This is not
// supported by the Intuos Wintab.  Your context will always receive
// packets, even if there has been no change in the data.
	logContext.lcMoveMask = PACKETDATA;

// Which buttons events will be handled by this context.  lcBtnMask
// is a bitfield with one bit per button.
	logContext.lcBtnUpMask = logContext.lcBtnDnMask;

// Set the entire tablet as active
	wWTInfoRetVal = gpWTInfoA( WTI_DEVICES, DVC_X, &TabletX );
	WACOM_ASSERT( wWTInfoRetVal == sizeof( AXIS ) );

	wWTInfoRetVal = gpWTInfoA( WTI_DEVICES, DVC_Y, &TabletY );
	WACOM_ASSERT( wWTInfoRetVal == sizeof( AXIS ) );

	logContext.lcInOrgX = 0;
	logContext.lcInOrgY = 0;
	logContext.lcInExtX = TabletX.axMax;
	logContext.lcInExtY = TabletY.axMax;

// Guarantee the output coordinate space to be in screen coordinates.  
// (It may not be for non-Intuos Wintabs).
	logContext.lcOutOrgX = logContext.lcOutOrgY = 0;
	logContext.lcOutExtX = GetSystemMetrics( SM_CXSCREEN );

// In Wintab, the tablet origin is lower left.  Move origin to upper left
// so that it coincides with screen origin.
	logContext.lcOutExtY = -GetSystemMetrics( SM_CYSCREEN );

	for ( wCurContext = 0 ;	wCurContext < NUMBER_WINTAB_CONTEXTS ; 
		wCurContext++ )
	{
	//	Note, you could also have a different hWnd for 
	// each context in order.  This would make handling the WT_PROXIMITY
	// and WT_CSRCHANGE messages easier.

		switch( wCurContext )
		{
			case DOMINANT_HAND_CONTEXT_INDEX:

				wsprintf( logContext.lcName, "CsrMask Dominant : %x", 
					ghProgramInstance );

				WACOM_TRACE( logContext.lcName );

			// Dominant cursor moves the system pointer.
				logContext.lcOptions |= CXO_SYSTEM;

			break;

			case NONDOMINANT_HAND_CONTEXT_INDEX:

				wsprintf( logContext.lcName, "CsrMask Nondominant : %x", 
					ghProgramInstance );

				WACOM_TRACE( logContext.lcName );

			// Nondominant context and cursor don't move the system pointer
				logContext.lcOptions &= ~CXO_SYSTEM;

			break;

			default:
				WACOM_ASSERT( 0 );
			break;
		}

		gWTContextInfo[ wCurContext ].eWintabCursor = EWINTAB_CURSOR_INDEX_NONE;

	// The Wintab spec says we must open the context disabled if we are 
	// using cursor masks.  
		gWTContextInfo[ wCurContext ].hCtx = gpWTOpenA( hWnd, &logContext, FALSE );
		
		WACOM_ASSERT( gWTContextInfo[ wCurContext ].hCtx );
		if ( !gWTContextInfo[ wCurContext ].hCtx )
		{
			return FALSE;
		}

		WACOM_TRACE( "%x WTOpen\n", gWTContextInfo[ wCurContext ].hCtx );

		bResult = gpWTQueueSizeSet( gWTContextInfo[ wCurContext ].hCtx, 
			WINTAB_CTX_QUEUE_SIZE );
		WACOM_ASSERT( bResult );
		if ( !bResult )
		{
			return FALSE;
		}

		// Get the initial, default cursor mask
		WACOM_TRACE("Get default cursor mask for: gWTContextInfo[ %i ].hCtx = %i", 
			wCurContext, gWTContextInfo[ wCurContext ].hCtx);

		bResult = gpWTExtGet( gWTContextInfo[ wCurContext ].hCtx, WTX_CSRMASK, 
			&gWTContextInfo[ wCurContext ].cCursorMask );

		WACOM_ASSERT( bResult );
		if ( !bResult )
		{
			return FALSE;
		}
	
		switch( wCurContext )
		{
			case DOMINANT_HAND_CONTEXT_INDEX:
		
			// Starting state is dominant context handles no cursors.  A cursor
			// will be added to the dominant context if the nondominant context
			// detects a dominant cursor.
				gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].cCursorMask[ 0 ] = 
					CSR_MASK_NO_CURSORS;

				gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].bIsDominant = TRUE;
				
			break;

			case NONDOMINANT_HAND_CONTEXT_INDEX:

			// Starting state is nondominant context handles all cursors.  If
			// the dominant context sees a dominant cursor enter the proximity
			// of the dominant context, we change the cursor mask of the dominant
			// context to handle the new dominant cursor.
				gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].cCursorMask[ 0 ] = 
					CSR_MASK_ALL_CURSORS;

				gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].bIsDominant = FALSE;

			break;

			default:
				WACOM_ASSERT( 0 );
			break;
		}

		bResult = gpWTExtSet( gWTContextInfo[ wCurContext ].hCtx, WTX_CSRMASK, 
			&gWTContextInfo[ wCurContext ].cCursorMask[ 0 ] );

		WACOM_ASSERT( bResult );
		if ( !bResult )
		{
			return FALSE;
		}

	// Enable context so we start getting messages.
		bResult = gpWTEnable( gWTContextInfo[ wCurContext ].hCtx, TRUE );
		WACOM_ASSERT( bResult );
	}

	return TRUE;
}


//////////////////////////////////////////////////////////////////////////////

LRESULT FAR PASCAL MainWndProc( HWND hWnd, unsigned wMessage, WPARAM wParam,
	LPARAM lParam )
{
	HDC hDC;
	PAINTSTRUCT	psPaint;
	BOOL bHandled;
	BOOL bResult;
	LRESULT lResult;

// When you call Wintab functions, there is a task switch.  In some cases,
// such as your WndProc processing takes a long time, your WndProc can  
// go reentrant.  bInWndProc tests for reentrancy.  If reentrancy is a problem
// for you, you need to take measures to account for it, such as protecting
// global data with critical sections and coding for cases where the
// Wintab packet queue is effected by reentrant code.

	static BOOL bInWndProc = FALSE;

	UINT wCurContext;
	UINT wInitResult;

	bHandled = TRUE;
	lResult = 0L;

	switch ( wMessage ) 
	{
		case WM_COMMAND:
				
			switch( LOWORD( wParam ) )
			{
				case IDM_ABOUT:

					WACOM_ASSERT( ghProgramInstance );

					DialogBox( ghProgramInstance, MAKEINTRESOURCE( IDD_ABOUT ), 
						hWnd, DlgAboutProc );
	
				break;

				case IDM_FILE_EXIT:

					DestroyWindow( hWnd );

				default:
				break;
			}
		break;

		case WM_CREATE:

			wInitResult = TabletInit( hWnd );
			if ( !wInitResult ) 
			{
				ShowError( "Could not open Wintab contexts." );
				SendMessage( hWnd, WM_DESTROY, 0, 0L );
			}

		break;

		case WM_SIZE:

			InvalidateRect( hWnd, NULL, TRUE );

		break;

		case WM_ACTIVATE:
		{
		// Wintab maintains a z-order of contexts.  In general, only the
		// topmost context receives tablet data.  When your application
		// receives WM_ACTIVATE because the user switched away from your 
		//	application, you must bring your context to the bottom of 
		// the z-order to help guarantee that any other Wintab applications 
		//	will receive tablet data while they are the active application.
		//
		// If your application receives a WM_ACTIVATE because the user 
		// switched TO your application, bring your context to the top
		// of the context z-order so that you receive tablet data while
		// your application is active.
		//
		// Note, the Intuos Wintab will automatically manage the z-order 
		// for you as is described here, but other Wintabs do not.

			WACOM_TRACE( ">>WM_ACTIVATE: CTX : %x %x\n", lParam, wParam );

			WACOM_ASSERT( gpWTEnable );
			WACOM_ASSERT( gpWTOverlap );

		// If message is to deactivate our window
			if ( !LOWORD( wParam ) )
			{

			// Disable context first in this order so that we don't get
			// unexpected WT_CSRCHANGE messages
				WACOM_TRACE("Deactivate NONDOMINANT_HAND_CONTEXT_INDEX ...\n");
				gpWTEnable( gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hCtx, 
					FALSE );
				WACOM_TRACE("Deactivate DOMINANT_HAND_CONTEXT_INDEX ...\n");
				gpWTEnable( gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hCtx, 
					FALSE );

				WACOM_TRACE("Send NONDOMINANT_HAND_CONTEXT_INDEX to bottom ...\n");
				gpWTOverlap( gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hCtx, 
					FALSE );
				WACOM_TRACE("Send DOMINANT_HAND_CONTEXT_INDEX to bottom ...\n");
				gpWTOverlap( gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hCtx, 
					FALSE );

				gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor = 
					EWINTAB_CURSOR_INDEX_NONE;
				
				gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor = 
					EWINTAB_CURSOR_INDEX_NONE;
			}
			else
			{
			// else message is our window is being activated

			// NOTE: If you have two cursors of the same type (2 pens or 2 4D 
			// mice) in prox, you switch apps, then switch back, the dominant and
			// nondominant cursor may be switched from before the app switch. 
			// This happens because for two dominant or two nondominant cursors,
			// which one controls the system pointer just depends on which cursor
			// enters the context first, which you can't predict if both are
			// in physical proximity of the tablet and enter the logical proximity
			// of the context simultaneously.  If you want to fix this, you will
			// have to do something like remember the CSR_PHYSID and CSR_TYPE of
			// the dominant cursor before the app switch, then make that cursor
			// dominant when the user switches back to your app.
				WACOM_TRACE("ACTIVATE WINDOW...\n");

				WACOM_TRACE("Activate NONDOMINANT_HAND_CONTEXT_INDEX ...\n");
				gpWTOverlap( gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hCtx, 
					TRUE );
				WACOM_TRACE("Activate DOMINANT_HAND_CONTEXT_INDEX ...\n");
				gpWTOverlap( gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hCtx, 
					TRUE );

				WACOM_TRACE("Send DOMINANT_HAND_CONTEXT_INDEX to top ...\n");
				gpWTEnable( gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hCtx, 
					TRUE );
				WACOM_TRACE("Send NONDOMINANT_HAND_CONTEXT_INDEX to top ...\n");
				gpWTEnable( gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hCtx, 
					TRUE );
			}
				
			WACOM_TRACE( "<< WM_ACTIVATE\n" );

			bHandled = FALSE;
		}
		break;

		case WM_DESTROY:

		// If application is shutting down, close our contexts
			for ( wCurContext = 0 ; wCurContext < NUMBER_WINTAB_CONTEXTS ;	
				wCurContext++ )
			{
				if ( gWTContextInfo[ wCurContext ].hCtx )
				{
					gpWTClose( gWTContextInfo[ wCurContext ].hCtx );
					gWTContextInfo[ wCurContext ].hCtx = NULL;
				}
			}
												
			PostQuitMessage( 0 );

		break;

		case WM_PAINT:	

			hDC = BeginPaint( hWnd, &psPaint );
			if ( !hDC )
			{
				break;
			}
	
			for(	wCurContext = 0 ; wCurContext < NUMBER_WINTAB_CONTEXTS ; 
				wCurContext++ )
			{
				ShowWintabCursor( hWnd, hDC, 	&gWTContextInfo[ wCurContext ] );
			}

			ShowCursorDescriptions( hWnd, hDC );

			EndPaint( hWnd, &psPaint );

		break;

	// a tablet data packet has been received
		case WT_PACKET:
			if ( !bInWndProc )
			{
				bInWndProc = TRUE;
			
				WACOM_TRACE( ">>WT_PACKET: CTX : %x %x\n", lParam, wParam );

				SavePacket( ( HCTX ) lParam, wParam );
				InvalidateRect( hWnd, NULL, TRUE );

				WACOM_TRACE( "<<WT_PACKET\n" );

				bInWndProc = FALSE;
			}

		break;

	// This message is sent when a cursor enters a context's proximity.  Note
	// There is very little utility in using this message.  You could just
	// watch the pkCursor field of the packets to see what cursors enter
	// the context.

		case WT_CSRCHANGE:
			if ( ! bInWndProc )
			{
				bInWndProc = TRUE;

				WACOM_TRACE( ">>WT_CSRCHANGE : %x %x\n", lParam, wParam );

				bResult = HandleWTCursorChange( hWnd, wParam, lParam );
				InvalidateRect( hWnd, NULL, TRUE );

				WACOM_ASSERT( "<<WT_CSRCHANGE\n" );

				bInWndProc = FALSE;
			}

		break;

	// This message is sent when a cursor has entered or exited proximity
	// of a context.  You can't tell from the message WHICH cursor entered
	// or exited, which is a major problem and limitation of Wintab.

		case WT_PROXIMITY:
			if ( ! bInWndProc )
			{
				bInWndProc = TRUE;

				// A hard proximity event is one where the cursor is physically
				// entering or leaving the proximity of the tablet.  A soft proximity 
				//	event is when a cursor is entering or your context because of
				// a change in the context z-order, etc.
				WACOM_TRACE( ">>WT_PROXIMITY: %s  %s %x\n", 
					LOWORD( lParam ) ? "Entering" : "Leaving",
					HIWORD( lParam ) ? "Hard" : "Soft",
					wParam );

				HandleWTProximity( hWnd, wParam, lParam );
				InvalidateRect( hWnd, NULL, TRUE );

				WACOM_TRACE( "<<WT_PROXIMITY\n" );

				bInWndProc = FALSE;
			}

			break;

		default:
			bHandled = FALSE;
		break;
	}

	if ( bHandled )
	{
		return lResult;
	}

	return DefWindowProc( hWnd, wMessage, wParam, lParam );
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Handles the Wintab WT_CSRCHANGE message, which is sent when a new
//		cursor enters or leaves proximity of your context.
//	Params
//		hWnd - Window with Window Proc which is receiving messages for our
//			context(s).
//		wPacketNumber - Tablet packet serial number for the cursor which
//			entered proximity of our context.
//		hCtxIn - Wintab context where WT_CSRCHANGE message occurred.
//
BOOL HandleWTCursorChange( HWND hWnd, WPARAM wPacketNumber, LPARAM hCtxIn )
{
	UINT wNewDominantCursor;

	HCTX hCtx;

	UINT wCursorType;
	UINT wCursorPhysicalID;
	UINT wWTInfoRetVal;

	BOOL bNeedToChangeDominantCursor;

	PACKET packet;

	BOOL bRetVal;

	int nNumPacketsCopied;

	WACOM_TRACE( "HandleWTCursorChange()\n" );

	WACOM_ASSERT( hCtxIn );
	WACOM_ASSERT( gpWTDataPeek );
	WACOM_ASSERT( gpWTInfoA );

	hCtx = ( HCTX ) hCtxIn;

	bRetVal = FALSE;

	if ( hCtx == gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hCtx )
	{
		WACOM_TRACE( "Wrong CTX for WT_CSRCHANGE\n" );
		return TRUE;
	}

	nNumPacketsCopied = 0;

	// peek -- don't remove first packet with the new cursor

	if ( !gpWTDataPeek( hCtx, wPacketNumber, wPacketNumber, 1, 
		&packet, &nNumPacketsCopied ) )
	{
		//WACOM_TRACE( "WTDataPeek failed\n" );
		//WACOM_ASSERT( 0 );

		return FALSE;
	}

	if ( nNumPacketsCopied != 1 )
	{
		WACOM_ASSERT( 0 );
		return FALSE;
	}

	WACOM_ASSERT( packet.pkCursor <= MAX_EWINTAB_CURSOR_INDEX );

	// Get CSR_TYPE and CSR_PHYSID info

	wWTInfoRetVal = gpWTInfoA( WTI_CURSORS + packet.pkCursor,
		CSR_TYPE, &wCursorType );
	WACOM_ASSERT( wWTInfoRetVal == sizeof( UINT ) );

	wWTInfoRetVal = gpWTInfoA( WTI_CURSORS + packet.pkCursor,
		CSR_PHYSID, &wCursorPhysicalID );
	WACOM_ASSERT( wWTInfoRetVal == sizeof( UINT ) );

	bNeedToChangeDominantCursor = FALSE;

	// Decide if the new cursor should move the screen pointer.  This
	// is based on the algorithm described at the top of this module.

	if ( gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor == 
		EWINTAB_CURSOR_INDEX_NONE )
	{
	// If dominant hand context is empty, then new cursor moves the
	// system pointer

		WACOM_TRACE( "No cursor controls the system cursor.\n" );
		WACOM_TRACE( "Setting new cursor to dominant context.\n" );

		wNewDominantCursor = packet.pkCursor;
		bNeedToChangeDominantCursor = TRUE;

		gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor = 
			EWINTAB_CURSOR_INDEX_NONE;

		gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].wCursorType = 
			wCursorType;

		gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].wCursorPhysicalID = 
			wCursorPhysicalID;
	}
	else
	{
	// else dominant hand context is already controlling a cursor

		if ( IsNonDominantCursor( gWTContextInfo[ 
			DOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor ) )
		{
			WACOM_TRACE( "A nondominant cursor is moving system cursor.\n" );

			if ( IsDominantCursor( packet.pkCursor ) )
			{
				WACOM_TRACE( "A new dominant cursor is being used by" 
					"the user.\n" );
				WACOM_TRACE( "Setting new dominant context to dominant"  
					"cursor.\n" );
				WACOM_TRACE( "Setting nondominant context to new nondominant"  
					"cursor.\n" );

				bNeedToChangeDominantCursor = TRUE;
				wNewDominantCursor = packet.pkCursor;

			// Newly enter pen moves the cursor. The other cursor doesn't
			// move the system cursor.

				gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor =
					gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor;

				gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].wCursorType = 
					gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].wCursorType;

				gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].wCursorPhysicalID = 
					gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].wCursorPhysicalID;

				gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].wCursorType = 
					wCursorType;

				gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].wCursorPhysicalID = 
					wCursorPhysicalID;
			}
			else
			{
			// A dominant cursor is already moving the system pointer, so
			// new cursor doesn't move the system cursor.
				gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor =
					packet.pkCursor;

				gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].wCursorType = 
					wCursorType;

				gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].
					wCursorPhysicalID = wCursorPhysicalID;

				WACOM_TRACE( "A new nondominant cursor is being used by" 
					"the user" );
				WACOM_TRACE( "Setting nondominant context to new nondominant"  
					"cursor.\n" );
			}
		}
		else
		{
		// A dominant cursor is already moving the system pointer, so
		// new cursor doesn't move the system cursor.
			gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor =
				packet.pkCursor;

			gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].wCursorType = 
				wCursorType;

			gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].
				wCursorPhysicalID = wCursorPhysicalID;

			WACOM_TRACE( "A dominant cursor is moving the system cursor.\n" );
			WACOM_TRACE( "Setting new cursor to nondominant context.\n" );
		}
	}

	if ( bNeedToChangeDominantCursor )
	{
		WACOM_TRACE( "Setting new dominant cursor to cursor %d.\n", 
			wNewDominantCursor );

		gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor = 
			wNewDominantCursor;

		SetCursorMask( hWnd, wNewDominantCursor );
	}

	bRetVal = TRUE;

	return bRetVal;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Handles WT_PROXIMITY messages, which are generated when a cursor
//		enters or leaves proximity of our contexts.  We are only interested
//		in WT_PROXIMITY messages for cursors leaving the dominant-hand context.
//
//		You cannot tell from this message which cursor entered or left proximity.
//
// Params
//		hWnd - Window with Window Proc which is receiving messages for our
//			context(s).
//		hCtxIn - Wintab context handle for the context where this event occurred.
//		dwTypeOfProximityChange - From LPARAM value of WT_PROXIMITY message.
//			It tells if the cursor entered or left the context and why.
//
BOOL HandleWTProximity( HWND hWnd, WPARAM hCtxIn, 
	LPARAM dwTypeOfProximityChange )
{
	BOOL bRetVal;

	WACOM_TRACE( "HandleWTProximity()\n" );

	WACOM_ASSERT( hWnd );
	WACOM_ASSERT( hCtxIn );
	WACOM_ASSERT( gpWTPacket );

	bRetVal = TRUE;

// We are only interested in leaving proximity in the dominant hand context. 
//	Entering proxes are handled by the WT_CSRCHANGE message handler because
// WT_CSRCHANGE tells us which cursor entered.

	if ( LOWORD( dwTypeOfProximityChange ) )
	{
		WACOM_TRACE( "Entering Prox - ignore message.\n" );
		return bRetVal;
	}

	if ( ( HCTX ) hCtxIn != gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hCtx )
	{
	// only interested in leaving messages from the dominant hand context

		WACOM_TRACE( "Wrong context for WT_PROXIMITY\n");
		
		gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor = 
			EWINTAB_CURSOR_INDEX_NONE;

		return bRetVal;
	}

	WACOM_ASSERT( ( HCTX ) hCtxIn == gWTContextInfo[ 
		DOMINANT_HAND_CONTEXT_INDEX ].hCtx );

	WACOM_TRACE( "Dominant cursor is leaving the dominant context.\n" );

	gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor = 
		EWINTAB_CURSOR_INDEX_NONE;

	SetCursorMask( hWnd,	
		gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].eWintabCursor );

// Empty the packet queue
	gpWTPacket( ( HCTX ) hCtxIn, 0xFFFF, NULL );

	return bRetVal;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		This code is called in response to the WT_PACKET message received by
//		one of our contexts.  We read the packet from the Wintab context's 
//		packet queue and save the packet data in which we are interested.
// Params
//		hCtx - Wintab context handle for the context where this event occurred.
//		wPacketNumber - The packet's serial number (same as pkSerialNumber
//		of the packet).
//
UINT SavePacket( HCTX hCtx, UINT wPacketNumber )
{
	UINT wCurContext;

	PACKET packet;

	static int nErrorCount;

//	WACOM_TRACE( "SavePacket()\n" );

	WACOM_ASSERT( hCtx );
	WACOM_ASSERT( gpWTPacket );

// Find Wintab Context information which corresponds to hCtx
	for(	wCurContext = 0 ; wCurContext < NUMBER_WINTAB_CONTEXTS ;	
		wCurContext++ )
	{
		if ( ( HCTX ) hCtx == gWTContextInfo[ wCurContext ].hCtx )
		{
			break;
		}
	}

	WACOM_ASSERT( wCurContext < NUMBER_WINTAB_CONTEXTS );

// Note, you can easily get failures here because WT_PACKET messages might be
// in your message queue while WM_ACTIVATE or WT_PROXIMITY messages are
// received.  The WT_PROXIMITY message handler flushes the packet queue

	if ( !gpWTPacket( hCtx, wPacketNumber, &packet ) )
	{
		WACOM_TRACE( "WTPacket Failed: %ld\n", nErrorCount++ );

		return FALSE;
	}

	gWTContextInfo[ wCurContext ].wCurX = packet.pkX;
	gWTContextInfo[ wCurContext ].wCurY = packet.pkY;
	gWTContextInfo[ wCurContext ].wCurZ = packet.pkZ;

	gWTContextInfo[ wCurContext ].wCurPressure = 
		packet.pkNormalPressure;
	gWTContextInfo[ wCurContext ].curOrientation = 
		packet.pkOrientation;

	return TRUE;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Displays data for the Wintab cursor.  There is only one cursor in
//		each of the two Wintab contexts we opened. 
//	Params
//		hWnd - Client window on which we will paint.
//		hDC - Device context of that window.
//		pCtxInfo - Cursor and context data to paint.
//
UINT ShowWintabCursor( HWND hWnd, HDC hDC, WINTAB_CONTEXT_INFO *pCtxInfo )
{
	int	ZAngle;			// Raw Altitude
	int	prs;

	int	lAdjustedZ;		// Adjusted Z coordinate

	UINT	Thata;			// Raw Azimuth
	UINT	Rot;				// Raw Rotation

	double ZAngle2;		// Adjusted Altitude
	double Thata2;			// Adjusted Azimuth
	double Rot2;			// Adjusted Rotation

	POINT Z1Angle;			// Rect coords from polar coords
	POINT Z2Angle;			// Rect coords for rotation

	POINT clientCoords;

	HGDIOBJ hPrevPen;

	HBRUSH hNewBrush;
	HBRUSH hPrevBrush;

	int adjX;
	int adjY;

//	WACOM_TRACE( "ShowWintabCursor()\n" );

	WACOM_ASSERT( hWnd );
	WACOM_ASSERT( IsWindow( hWnd ) );
	WACOM_ASSERT( hDC );
	WACOM_ASSERT( pCtxInfo );

	if ( pCtxInfo->eWintabCursor == EWINTAB_CURSOR_INDEX_NONE )
	{
	// if the context currently doesn't have a cursor (a cursor
	// is not in proximity of the tablet), we can't display it.
		return TRUE;
	}

	if ( tilt_support ) 
	{
	//	wintab.h defines .orAltitude 
	//	as a UINT but documents .orAltitude 
	//	as positive for upward angles 
	//	and negative for downward angles.
	//	WACOM uses negative altitude values to 
	//	show that the pen is inverted; 
	//	therefore we cast .orAltitude as an 
	//	(int) and then use the absolute value. 

		ZAngle  = ( int ) pCtxInfo->curOrientation.orAltitude;
		ZAngle2 = altAdjust - ( double ) abs( ZAngle ) / altFactor;

	// adjust azimuth
		Thata  = pCtxInfo->curOrientation.orAzimuth;
		Thata2 = ( double ) Thata / aziFactor;

	// get the length of the diagnal to draw
		Z1Angle.x = ( int ) ( ZAngle2 * sin( Thata2 ) );
		Z1Angle.y = ( int ) ( ZAngle2 * cos( Thata2 ) );
	}
	else 
	{
		ZAngle = 0;
		Thata = 0;
		Z1Angle.x = 0;
		Z1Angle.y = 0;
	}

	if ( rot_support ) 
	{
		WACOM_ASSERT( rotFactor );

	// adjust rotation
		Rot  = pCtxInfo->curOrientation.orTwist;
		Rot2 = ( 2 * pi ) - ( double ) Rot / rotFactor;

	// get the diagnal to draw
		Z2Angle.x = ( int ) ( 95 * sin( Rot2 ) );
		Z2Angle.y = ( int ) ( 95 * cos( Rot2 ) );
	}
	else 
	{
		Rot = 0;
		Z2Angle.x = 0;
		Z2Angle.y = 0;
	}

	hPrevPen = SelectObject( hDC,	pCtxInfo->hPen );

	WACOM_ASSERT( prsFactor );

	prs = ( int )( ( double ) pCtxInfo->wCurPressure / prsFactor );

	clientCoords.x = pCtxInfo->wCurX;
	clientCoords.y = pCtxInfo->wCurY;

	ScreenToClient( hWnd, &clientCoords );

	adjX = clientCoords.x;
	adjY = clientCoords.y;

// Draw circle based on tablet pressure
	Ellipse( hDC, adjX - prs, adjY - prs, adjX + prs, adjY + prs );

// Currently stylii don't have rotation and pucks don't have tilt, but this
// could change in the future.

// Draw a line based on tablet tilt

	if ( IsPen( pCtxInfo->eWintabCursor ) )
	{ 
		MoveToEx( hDC, adjX, adjY, NULL );
		LineTo(	hDC, adjX + Z1Angle.x, adjY - Z1Angle.y );
	}
	else
	{
		if ( ( pCtxInfo->wCursorType & CSR_TYPE_SPECIFIC_MASK ) == 
			CSR_TYPE_SPECIFIC_4DMOUSE_BITS )
		{
		// Draw a line based on 4D mouse rotation

			MoveToEx( hDC, adjX, adjY, NULL );
			LineTo(	hDC, adjX + ( Z2Angle.x / 2 ), adjY - ( Z2Angle.y / 2 ) );

			MoveToEx( hDC, adjX, adjY, NULL );
			LineTo(	hDC, adjX - ( Z2Angle.x / 2 ), adjY + ( Z2Angle.y / 2 ) );

		// Draw circle based on z-coordinate.  Note that currently this data
		// item is from the thumbwheel of pucks.  The default is for the 
		// thumbwheel to control window scrolling, not give z-coordinates.
		// If you want z-coordinates, you must set the thumbwheel output
		// to application defined in the Intuos control panel.  There is 
		//	currently no way to set this from your program.

			if( pCtxInfo->wCurZ )
			{
				adjX = clientCoords.x;
				adjY = clientCoords.y;

			// Thumbwheel returns data +-1023

				lAdjustedZ = ( int ) pCtxInfo->wCurZ / 10;

				hNewBrush = NULL;
				hPrevBrush = NULL;

				if ( lAdjustedZ < 0 )
				{
					hNewBrush = CreateSolidBrush( RGB( 0, 0, 255 ) );
					if ( hNewBrush )
					{
						hPrevBrush = SelectObject( hDC, hNewBrush );
					}
				}

				lAdjustedZ = abs( lAdjustedZ );

				Ellipse( hDC, adjX - lAdjustedZ, adjY - lAdjustedZ, 
					adjX + lAdjustedZ, adjY + lAdjustedZ );

				if ( hPrevBrush )
				{
					SelectObject( hDC, hPrevBrush );
				}

				if ( hNewBrush )
				{
					DeleteObject( hNewBrush );
				}
			} // if ( pCtxInfo->wCurZ )
		} // if ( 4DMouse )
	} // else not a pen
				
// draw CROSS based on tablet position
	MoveToEx( hDC, adjX - 20, adjY, NULL );
	LineTo( hDC, adjX + 20, adjY );

	MoveToEx( hDC, adjX, adjY - 20, NULL );
	LineTo( hDC, adjX, adjY + 20 );

	SelectObject( hDC, hPrevPen );

	return TRUE;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Displays text information about inprox Wintab cursors.
//
void ShowCursorDescriptions( HWND hWnd, HDC hDC )
{
	char szBuff[ 50 ];

	char *pszGeneralCursorName;
	char *pszSpecificCursorName;

// 0-5 id for cursor
	UINT wWintabCursorIndex; 

// WTInfo CSR_TYPE
	UINT wCursorType;

	COLORREF prevColor;

	int nOldBkMode;

	int nLineNumber;

	TEXTMETRIC textMetrics;

	UINT wTextHeight;
	UINT wCurContext;

	WACOM_TRACE( "ShowCursorDescriptions()\n" );

	WACOM_ASSERT( hWnd );
	WACOM_ASSERT( hDC );

	GetTextMetrics( hDC, &textMetrics );

	wTextHeight = textMetrics.tmHeight + textMetrics.tmExternalLeading;

	nOldBkMode = SetBkMode( hDC, TRANSPARENT );

	nLineNumber = 0;

	for( wCurContext = 0; wCurContext < NUMBER_WINTAB_CONTEXTS ;
		wCurContext++ )
	{
		if ( gWTContextInfo[ wCurContext ].eWintabCursor == 
			EWINTAB_CURSOR_INDEX_NONE )
		{
		// If no cursor is in proximity in this context, nothing to do
			continue;
		}
	
		prevColor = SetTextColor( 
			hDC, 
			wCurContext == DOMINANT_HAND_CONTEXT_INDEX ? 
				DOMINANT_COLOR : NONDOMINANT_COLOR );

		wWintabCursorIndex = gWTContextInfo[ wCurContext ].eWintabCursor;

		pszGeneralCursorName = GetGeneralCursorName( wWintabCursorIndex );

		WACOM_ASSERT( pszGeneralCursorName );
		if( pszGeneralCursorName )
		{
			sprintf( szBuff, "Cursor Index: %d (%s)", wWintabCursorIndex,
				pszGeneralCursorName );

			TextOut( hDC, 0, wTextHeight * nLineNumber++, szBuff, 
				lstrlen( szBuff ) );
		}

		wCursorType = gWTContextInfo[ wCurContext ].wCursorType;

		pszSpecificCursorName = GetSpecificCursorName( wCursorType );

		WACOM_ASSERT( pszSpecificCursorName );
		if ( pszSpecificCursorName )
		{
			sprintf( szBuff, "Cursor Type: %X (%s)", wCursorType,
				pszSpecificCursorName );

			TextOut( hDC, 0, wTextHeight * nLineNumber++, szBuff, 
				lstrlen( szBuff ) );
		}

		sprintf( szBuff, "PhysicalID: %X", 
			gWTContextInfo[ wCurContext ].wCursorPhysicalID );

		TextOut( hDC, 0, wTextHeight * nLineNumber++, szBuff, 
			lstrlen( szBuff ) );

		if ( prevColor )
		{
			SetTextColor( hDC, prevColor );
		}
	}

	if ( nOldBkMode )
	{
		SetBkMode( hDC, nOldBkMode );
	}
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Standard DlgProc for the about dialog.
//
BOOL FAR PASCAL DlgAboutProc( HWND hDlg, unsigned wMessage, WPARAM wParam,
	LPARAM lParam )
{
	BOOL bRetVal;

	bRetVal = FALSE;

	switch ( wMessage ) 
	{
		case WM_INITDIALOG:

			bRetVal = TRUE;
			break;

		case WM_COMMAND:
			
			if ( LOWORD( wParam ) == IDOK )
			{
				EndDialog( hDlg, TRUE );
				bRetVal = TRUE;
			}
			break;

		default:
		break;
	}

	return bRetVal;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Disables our contexts, sets the new cursor mask for the dominant
//		context, then reenables our contexts.
// Params
//		hWnd - Window handle for the Window with the Window Procedure which
//			is handling Wintab messages for our contexts.
//		wNewDominantCursor - Wintab cursor index (0-5) for the cursor which will
//			now control the system pointer.
//
BOOL SetCursorMask( HWND hWnd, DWORD wNewDominantCursor )
{
	MSG msg;

	BOOL bResult;

	WACOM_TRACE( "SetCursorMask()\n" );

	WACOM_ASSERT( hWnd );
	WACOM_ASSERT( ( wNewDominantCursor < 5 ) || 
		( wNewDominantCursor == EWINTAB_CURSOR_INDEX_NONE ) );

	WACOM_TRACE( "Setting dominant context to Cursor %d\n", 
		wNewDominantCursor );

	if ( wNewDominantCursor == EWINTAB_CURSOR_INDEX_NONE )
	{
		gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].cCursorMask[ 0 ] = 0;
	}
	else
	{
	// Convert the cursor id (index) into a bit in the cursormask
	// bit field.
		gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].cCursorMask[ 0 ] = 
			1 << wNewDominantCursor;
	}

// Disable contexts so that we can set the cursor mask.

	WACOM_TRACE( "Disabling Contexts\n" );

	bResult = gpWTEnable( 
		gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hCtx, FALSE );
	WACOM_ASSERT( bResult );

	bResult = gpWTEnable( 
		gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hCtx, FALSE );
	WACOM_ASSERT( bResult );

// We will get a WT_PROXIMITY(Leaving) message in the dominant context
// when we disable the contexts here.  We need to avoid this in order
// to keep track of which cursor is in the dominant context. 

// NOTE! If two cursors exit prox at the same time, the WT_PROXIMITY
// for the dominant cursor will get eaten here and you'll miss the
// WT_PROXIMITY(leaving).  A refinement which solves this and similar
// problems is to use separate window handles for the two contexts.
// You would then just remove the WT_PROXIMITIES for the dominant
// context.

	Sleep( 50 );

// Remove all WT_PROXIMITY messages from queue
	while ( PeekMessage( &msg, hWnd, WT_PROXIMITY, WT_PROXIMITY, PM_REMOVE ) )
	{
		WACOM_TRACE( "Removing WT_Message Before Enable\n" );
	}

	bResult = gpWTExtSet( 
		gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hCtx, 
		WTX_CSRMASK, 
		&gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].cCursorMask[ 0 ] );
	WACOM_ASSERT( bResult );

	bResult = gpWTEnable( 
		gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hCtx, TRUE );
	WACOM_ASSERT( bResult );

	bResult = gpWTEnable( 
		gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hCtx, TRUE );
	WACOM_ASSERT( bResult );

	Sleep( 50 );

	while ( PeekMessage( &msg, hWnd, WT_PROXIMITY, WT_PROXIMITY, PM_REMOVE) )
	{
		WACOM_TRACE( "Removing WT_Message After Enable\n" );
	}

	WACOM_TRACE( "Contexts Enabled\n" );

	return TRUE;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Tests whether wCursor should move the system pointer.
//	Param
//		wCursor - Wintab cursor index (0-5).
// Returns
//		TRUE if wCursor should move the system cursor.
//		FALSE if wCursor should not move the system cursor.
//
BOOL IsDominantCursor( UINT wCursor )
{
	WACOM_TRACE( "IsDominantCursor()\n" );

	return IsNonDominantCursor( wCursor ) ? FALSE : TRUE;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Tests whether wCursor is a pen-like cursor (stylus, airbrush, inking
//		pen, strokepen, etc.)
//	Params
//		wCursor - Wintab cursor index (0-5)
// Returns
//		TRUE if wCursor is a pen-like cursor
//		FALSE if wCursor is not a pen-like cursor.
//
BOOL IsPen( UINT wCursor )
{
	WACOM_TRACE( "IsPen()\n" );

	return ( wCursor == EWINTAB_CURSOR_INDEX_STYLUS_0 ) ||
				( wCursor == EWINTAB_CURSOR_INDEX_STYLUS_1 ) ||
				( wCursor == EWINTAB_CURSOR_INDEX_STYLUS_INVERTED_0 ) || 
				( wCursor == EWINTAB_CURSOR_INDEX_STYLUS_INVERTED_1 );
}

//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Tests whether a specific cursor should not move the system cursor.
//	Params
//		wCursor - Wintab cursor index (0-5)
//	Returns
//		TRUE if wCursor should not move the system cursor.
//		FALSE if wCursor should move the system cursor.
//
BOOL IsNonDominantCursor( UINT wCursor )
{
	WACOM_TRACE( "IsNonDominantCursor()\n" );

	WACOM_ASSERT( wCursor != EWINTAB_CURSOR_INDEX_NONE );

	return( wCursor == EWINTAB_CURSOR_INDEX_PUCK_0 ) || 
			( wCursor == EWINTAB_CURSOR_INDEX_PUCK_1 ) ?	
				TRUE : FALSE;
}



//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Given a Wintab cursor index (0-5), returns a name generally describing
//		the cursor.
// Returns
//		Ptr to generic cursor name on success
//		NULL on error
//
char *GetGeneralCursorName( UINT wCursorIndex )
{
	char *pRetVal;

	WACOM_TRACE( "GetGeneralCursorName()\n" );

	pRetVal = NULL;

	switch( wCursorIndex )
	{
		case EWINTAB_CURSOR_INDEX_PUCK_0:
			pRetVal = "First Puck";
		break;

		case EWINTAB_CURSOR_INDEX_STYLUS_0:
			pRetVal = "First Pen Tip";
		break;

		case EWINTAB_CURSOR_INDEX_STYLUS_INVERTED_0:
			pRetVal = "First Eraser";
		break;

		case EWINTAB_CURSOR_INDEX_PUCK_1:
			pRetVal = "Second Puck";
		break;

		case EWINTAB_CURSOR_INDEX_STYLUS_1:
			pRetVal = "Second Pen Tip";
		break;

		case EWINTAB_CURSOR_INDEX_STYLUS_INVERTED_1:
			pRetVal = "Second Eraser";
		break;

		default:
			WACOM_ASSERT( 0 );
		break;
	}

	return pRetVal;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Given a CSR_TYPE returned from 
//		WTInfo( WTI_CURSORS + wCursorIndex, CSR_TYPE, returns a name which
//		identifies a Wacom Intuos cursor.
// Returns
//		Ptr Specific cursor name on success
//		NULL on error
//
char *GetSpecificCursorName( UINT wCursorType )
{
	char *pRetVal;

	WACOM_TRACE( "GetSpecificCursorName()\n" );

	pRetVal = NULL;

	switch( wCursorType & CSR_TYPE_SPECIFIC_MASK )
	{
		case CSR_TYPE_SPECIFIC_STYLUS_BITS:
			pRetVal = "Stylus";
		break;

		case CSR_TYPE_SPECIFIC_AIRBRUSH_BITS:
			pRetVal = "Airbrush";
		break;

		case CSR_TYPE_SPECIFIC_4DMOUSE_BITS:
			pRetVal = "4D Mouse";
		break;

		case CSR_TYPE_SPECIFIC_LENSCURSOR_BITS:
			pRetVal = "Lens Cursor";
		break;

		default:
		{
		// maybe it's a new cursor, so return more general
		// information.

			switch( wCursorType & CSR_TYPE_GENERAL_MASK )
			{
				case CSR_TYPE_GENERAL_PENTIP:
					pRetVal = "Pen Tip";
				break;

				case CSR_TYPE_GENERAL_PUCK:
					pRetVal = "Puck";
				break;

				case CSR_TYPE_GENERAL_PENERASER:
					pRetVal = "Eraser";
				break;

				default:
					WACOM_ASSERT( 0 );
				break;
			}
		}
		break;
	}

	return pRetVal;
}


//////////////////////////////////////////////////////////////////////////////
// Purpose
//		Release resources we used in this example.
//
void Cleanup( void )
{
	WACOM_TRACE( "Cleanup()\n" );

	if ( gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hPen )
	{
		DeleteObject( gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hPen );
		gWTContextInfo[ DOMINANT_HAND_CONTEXT_INDEX ].hPen = NULL;
	}

	if ( gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hPen )
	{
		DeleteObject( gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hPen );
		gWTContextInfo[ NONDOMINANT_HAND_CONTEXT_INDEX ].hPen = NULL;
	}

	UnloadWintab( );
}



//////////////////////////////////////////////////////////////////////////////

