/*----------------------------------------------------------------------------

	NAME
		Rule.c

	PURPOSE
		Pen button handling demo.

	COPYRIGHT
		Copyright (C) 1998  LCS/Telegraphics
		Copyright (c) Wacom Company, Ltd. 2014 All Rights Reserved
		All rights reserved.

		The text and information contained in this file may be freely used,
		copied, or distributed without compensation or licensing restrictions.

---------------------------------------------------------------------------- */
#include <windows.h>
#include <stdlib.h>
#include "Utils.h"
#include "MsgPack.h"
#include "CadTest.h"
#include "Rule.h"

#define Inch2Cm	CASTFIX32(2.54)
#define Cm2Inch	CASTFIX32(1.0/2.54)

/* local functions */
DWORD nsqrt(DWORD x);
HCTX TabletRuleInit(HWND hWnd, FIX32 scale[]);
void TabletRuleScaling(FIX32 scale[]);

/* -------------------------------------------------------------------------- */
BOOL FAR PASCAL RuleDemoProc(hDlg, message, wParam, lParam)
HWND hDlg;
unsigned message;
WPARAM wParam;
LPARAM lParam;
{
	static int inMode = ID_CLICK;
	static LONG x1 = 0, x2 = 0, y1 = 0, y2 = 0;
	static HCTX hTab = NULL;

	static FIX32 scale[2] = {0};

	PAINTSTRUCT psPaint;
	HDC hDC;

	switch (message) 
	{
		case WM_INITDIALOG:
			inMode = ID_CLICK;
			x1 = x2 = y1 = y2 = 0;

			TabletRuleScaling(scale);

			if ( !hTab )
			{
				hTab = TabletRuleInit(hDlg, scale);
			}
			return (TRUE);

		case WM_COMMAND:
			if (GET_WM_COMMAND_ID(wParam, lParam) == IDOK
					|| GET_WM_COMMAND_ID(wParam, lParam) == IDCANCEL) 
			{
				if ( hTab )
				{
					gpWTClose(hTab);
					hTab = NULL;
				}

				EndDialog(hDlg, TRUE);
				return (TRUE);
			}

			break;

		case WM_LBUTTONDOWN:
			if ( hTab )
			{
				PACKET pkt;
				BOOL bSawFirstButtonPkt = FALSE;

				inMode = ID_PRESS;
				x1 = x2 = y1 = y2 = 0;
				InvalidateRect(hDlg, NULL, TRUE);
				UpdateWindow(hDlg);

				while (inMode != ID_CLICK) 
				{
					/* poll */
					if (!gpWTPacketsGet(hTab, 1, &pkt))
					{
						continue;
					}

					// Sometimes, the first packet comes through with no button data.
					// If so, just ignore it and wait for the next packet.
					if ( ! bSawFirstButtonPkt )
					{
						if ( ! pkt.pkButtons )
						{
							//WACOM_TRACE("Disregarding packet w/no button data...\n");
							continue;	// disregard this packet
						}

						bSawFirstButtonPkt = TRUE;
					}

					//WACOM_TRACE("pkt: [%i,%i], pkt.pkButtons: %i\n", pkt.pkX, pkt.pkY, pkt.pkButtons);

					/* handle it */
					if (inMode == ID_PRESS && pkt.pkButtons) 
					{
						//WACOM_TRACE("PRESSING: pkt: [%i,%i]\n", pkt.pkX, pkt.pkY);
						x1 = pkt.pkX;
						y1 = pkt.pkY;
						inMode = ID_RELEASE;
						InvalidateRect(hDlg, NULL, TRUE);
						UpdateWindow(hDlg);
					}
					if (inMode == ID_RELEASE && !pkt.pkButtons) 
					{
						//WACOM_TRACE("RELEASED: pkt: [%i,%i]\n", pkt.pkX, pkt.pkY);
						x2 = pkt.pkX;
						y2 = pkt.pkY;
						inMode = ID_CLICK;
						InvalidateRect(hDlg, NULL, TRUE);
						UpdateWindow(hDlg);
					}
				}
			}
			break;

		case WM_PAINT:
			hDC = BeginPaint(hDlg, &psPaint);
			ShowWindow(GetDlgItem(hDlg, ID_CLICK), inMode == ID_CLICK);
			ShowWindow(GetDlgItem(hDlg, ID_PRESS), inMode == ID_PRESS);
			ShowWindow(GetDlgItem(hDlg, ID_RELEASE), inMode == ID_RELEASE);
			if (inMode == ID_CLICK || inMode == ID_PRESS) 
			{
				LONG delta[3];	/* horz/vert/diag */
				int i;

				delta[0] = labs(x2 - x1);
				delta[1] = labs(y2 - y1);
				delta[2] = nsqrt(delta[0] * delta[0] + delta[1] * delta[1]);

				for (i = 0; i < 3; i++) /* direction */
				{	 	
					char buf[20];

					/* print result in cm */
					wsprintf(buf, "%d.%3.3d", (UINT)delta[i]/1000,
						(UINT)delta[i]%1000);
					SetWindowText(GetDlgItem(hDlg, ID_HC + i), buf);

					/* convert to inches */
					delta[i] = INT(delta[i] * Cm2Inch);

					/* print result in inches */
					wsprintf(buf, "%d.%3.3d", (UINT)delta[i]/1000,
						(UINT)delta[i]%1000);
					SetWindowText(GetDlgItem(hDlg, ID_HI + i), buf);
				}
			}

			EndPaint(hDlg, &psPaint);
			break;
	}

	return (FALSE);
}


/* -------------------------------------------------------------------------- */
static HCTX TabletRuleInit(HWND hWnd, FIX32 scale[])
{
	HCTX hTab = NULL;
	LOGCONTEXT lcMine;

	/* get default region */
	gpWTInfoA(WTI_DEFCONTEXT, 0, &lcMine);

	/* modify the digitizing region */
	strcpy(lcMine.lcName, "Rule Digitizing");
	lcMine.lcPktData = PACKETDATA;
	lcMine.lcPktMode = PACKETMODE;
	lcMine.lcMoveMask = 0;
	lcMine.lcBtnUpMask = lcMine.lcBtnDnMask;

	/* output in 1000ths of cm */
	lcMine.lcOutOrgX = lcMine.lcOutOrgY = 0;
	lcMine.lcOutExtX = INT(scale[0] * lcMine.lcInExtX);
	lcMine.lcOutExtY = INT(scale[1] * lcMine.lcInExtY);

	/* open the region */
	hTab = gpWTOpenA(hWnd, &lcMine, TRUE);

	if (!hTab) {
		MessageBox(NULL, " Could Not Open Tablet Context.", gpszProgramName,
					MB_OK | MB_ICONHAND);

		SendMessage(hWnd, WM_DESTROY, 0, 0L);
	}

	return hTab;

}


/* -------------------------------------------------------------------------- */
/* return scaling factors in thousandths of cm per axis unit */
static void TabletRuleScaling(FIX32 scale[])
{
	AXIS aXY[2];
	int i;
	UINT wDevice;

	/* get the data */
	gpWTInfoA(WTI_DEFCONTEXT, CTX_DEVICE, &wDevice);
	gpWTInfoA(WTI_DEVICES+wDevice, DVC_X, &aXY[0]);
	gpWTInfoA(WTI_DEVICES+wDevice, DVC_Y, &aXY[1]);

	/* calculate the scaling factors */
	for (i = 0; i < 2; i++) {
		FIX_DIV(scale[i], CASTFIX32(1000), aXY[i].axResolution);
		if (aXY[i].axUnits == TU_INCHES) {
			FIX_MUL(scale[i], scale[i], Inch2Cm);
		}
	}
}


/* -------------------------------------------------------------------------- */
static DWORD nsqrt(DWORD x)
{
	/* integer square root via Newton's method. */
	DWORD guess, oguess;

	if (x <= 1)
		return x;

	guess = 1;
	do
	{
		oguess = guess;
		guess = (guess + x/guess)/2;
	}
	while (labs(guess - oguess) > 1);

	if (guess == oguess)
		guess++;

	if (labs((guess * guess) - x) > labs((oguess * oguess) - x))
		guess = oguess;

	return guess;
}
