/*----------------------------------------------------------------------------

 FILE NAME
 FingerDataView.m

 PURPOSE
 A view which displays finger data from the Wacom MultiTouch API.

 COPYRIGHT
 Copyright WACOM Technology, Inc.  2011.
 All rights reserved.

 ----------------------------------------------------------------------------*/
#import "FingerDataView.h"



//////////////////////////////////////////////////////////////////////////////
// CopyFingerCollection()
//
// Purpose:		Returns a malloc'd copy of the finger collection.
//
// Notes:		Since finger collections contain allocated pointers, we need a
//					special copy routine to ensure the memory stays valid. Structures
//					returned from the Wacom Touch API are only guaranteed to stay
//					valid for the lifetime of the callback.
//
static WacomMTFingerCollection *CopyFingerCollection(WacomMTFingerCollection *original)
{
	WacomMTFingerCollection  *copied  = calloc(1, sizeof(WacomMTFingerCollection));

	copied->Version      = original->Version;
	copied->DeviceID     = original->DeviceID;
	copied->FingerCount  = original->FingerCount;
	copied->Fingers      = calloc(original->FingerCount, sizeof(WacomMTFinger));

	// Copy all the finger structs
	memcpy(copied->Fingers, original->Fingers, sizeof(WacomMTFinger) * original->FingerCount);

	return copied;
}



//////////////////////////////////////////////////////////////////////////////
// FreeFingerCollection()
//
// Purpose:		Frees the memory pointed to by fingerCollection.
//
static void FreeFingerCollection(WacomMTFingerCollection *fingerCollection)
{
	// free allocated memory within the struct
	free(fingerCollection->Fingers);

	// free the struct itself.
	free(fingerCollection);
}


@implementation FingerDataView

#pragma mark -
#pragma mark INITIALIZATION
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// initWithFrame:
//
// Purpose:		Initialize the view.
//
- (id)initWithFrame:(NSRect)frame
{
	self = [super initWithFrame:frame];
	if (self)
	{
		self->fingerData           = [[NSMutableArray alloc] init];
		self->logicalSize          = NSZeroSize;
		self->maxPacketsToDisplay  = 1;
		displayMode = eDM_Panel;
	}
	return self;
}



//////////////////////////////////////////////////////////////////////////////
-(void) setDisplayMode:(eDisplayMode)mode
{
	displayMode = mode;
}


#pragma mark -
#pragma mark ACCESSORS
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// pushNewFingers:
//
// Purpose:		Adds a new set of finger data to the display.
//
- (void) pushNewFingers:(WacomMTFingerCollection)fingerCollection
{
	// You should not spend too long responding to touch messages. The touch API
	// will automatically and silently purge hung applications, requiring a
	// re-registration.
	//	usleep(0.02 * 1000000);

	// Erase all previous fingers?
	if(self->clearOnNextDown)
	{
		[self popAllFingers];
		self->clearOnNextDown = NO;
	}

	// Since we store the finger data locally for the sake of redrawing it, we
	// must make a copy of the data. The Wacom MultiTouch API only guarantees the
	// lifetime of the structs it passes to callbacks for the lifetime of the
	// function call itself.

	WacomMTFingerCollection *localCollectionCopy = CopyFingerCollection(&fingerCollection);
	NSValue                 *collectionPointer   = [NSValue valueWithPointer:localCollectionCopy];

	// Push latest data onto the packet stack
	[self->fingerData insertObject:collectionPointer atIndex:0];

	// Remove old packets
	if([self->fingerData count] > self->maxPacketsToDisplay)
	{
		NSValue *bumpedValue = [self->fingerData lastObject];
		FreeFingerCollection( (WacomMTFingerCollection*)[bumpedValue pointerValue] );
		[self->fingerData removeLastObject];
	}

	// Count all down fingers
	size_t   fingerCounter  = 0;
	size_t   fingerCount    = 0;
	for(fingerCounter = 0; fingerCounter < fingerCollection.FingerCount; fingerCounter++)
	{
		WacomMTFinger *finger = &fingerCollection.Fingers[fingerCounter];
		if(finger->TouchState == WMTFingerStateDown || finger->TouchState == WMTFingerStateHold)
		{
			fingerCount++;
		}
	}
	if(fingerCount == 0)
	{
		// All fingers lifted; make sure the display is wiped on next touch
		clearOnNextDown = YES;
	}

	[self setNeedsDisplay:YES];
}



//////////////////////////////////////////////////////////////////////////////
// setBoundsFromLogicalSize:
//
// Purpose:		Sets a bounds rect we maintain no matter what the size. By
//					setting the view bounds to the device coordinate space, we can
//					draw directly in device logical coordinates.
//
- (void) setBoundsFromLogicalSize:(NSSize)deviceSize
{
	if(displayMode == eDM_Panel)
	{
		[self setBoundsSize:deviceSize];
		self->logicalSize = deviceSize;
	}
}



//////////////////////////////////////////////////////////////////////////////
// setFrame:
//
// Purpose:		As the view is live resized, we force it to maintain the same
//					bounds rectangle (which has been set equal to device logical
//					coordinates).
//
- (void)setFrame:(NSRect)frameRect
{
	[super setFrame:frameRect];

	if(displayMode == eDM_Panel)
	{
		if(NSEqualSizes(self->logicalSize, NSZeroSize) == NO)
		{
			[self setBoundsSize:logicalSize];
		}
	}
}



#pragma mark -
#pragma mark DRAWING
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// drawRect:
//
// Purpose:		Plot all the finger coordinates.
//
- (void)drawRect:(NSRect)dirtyRect
{
	NSUInteger packetCounter = 0;

	[[NSColor blackColor] set];
	NSRectFill([self bounds]);

	for(packetCounter = 0; packetCounter < [self->fingerData count]; packetCounter++)
	{
		WacomMTFingerCollection *collection    = [[self->fingerData objectAtIndex:packetCounter] pointerValue];
		size_t                  fingerCounter  = 0;

		for(fingerCounter = 0; fingerCounter < collection->FingerCount; fingerCounter++)
		{
			// Draw the finger ONLY if it is down
			WacomMTFinger *finger = &collection->Fingers[fingerCounter];
			if(finger->TouchState == WMTFingerStateDown || finger->TouchState == WMTFingerStateHold)
			{
				NSBezierPath   *pointPath  = nil;
				NSRect         pointRect   = NSZeroRect;
				if(displayMode == eDM_Panel)
				{
					pointRect.origin  = NSMakePoint(finger->X , finger->Y);
				}
				else
				{
					//The coordinate systems are different in the window and the tablet. The tablet
					//coordinate system starts at the bottom left corner. The mac coordinate system
					//starts at the top right, therefore we have to convert between the two spaces using
					//the screen height and the frame height.
					pointRect.origin  = NSMakePoint(finger->X - self.window.frame.origin.x, finger->Y - (self.window.screen.frame.size.height - self.window.frame.size.height)+ self.window.frame.origin.y);
				}
				
				pointRect.size    = NSMakeSize(finger->Width * 4, finger->Height * 4); // expand to improve visibility

				pointRect = NSOffsetRect(pointRect, -NSWidth(pointRect)/2, -NSHeight(pointRect)/2);
				pointPath = [NSBezierPath bezierPathWithOvalInRect:pointRect];

				[[self colorForFinger:fingerCounter] set];
				[pointPath fill];
			}
		}
	}
}



//////////////////////////////////////////////////////////////////////////////
// isFlipped
//
// Purpose:		The tablet coordinate system has (0,0) in the upper-left. Cocoa's
//					coordinate system has the origin in the lower-left by default,
//					but this method allows us to use the tablet's system for drawing.
//
- (BOOL) isFlipped
{
	return YES;
}



//////////////////////////////////////////////////////////////////////////////
// colorForFinger:
//
// Purpose:		Hard-codes a table of colors to distinguish fingers while
//					drawing.
//
- (NSColor *) colorForFinger:(size_t)fingerIndex
{
	NSColor *color = nil;

	switch(fingerIndex)
	{
		case 0:	color = [NSColor redColor];		break;
		case 1:	color = [NSColor greenColor];		break;
		case 2:	color = [NSColor blueColor];		break;
		case 3:	color = [NSColor cyanColor];		break;
		case 4:	color = [NSColor magentaColor];	break;
		case 5:	color = [NSColor yellowColor];	break;
		case 6:	color = [NSColor grayColor];		break;
		case 7:	color = [NSColor orangeColor];	break;
		case 8:	color = [NSColor purpleColor];	break;
		case 9:	color = [NSColor brownColor];		break;

		default: color = [NSColor whiteColor];		break;
	}
	return color;
}



#pragma mark -
#pragma mark UTILITIES
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// popAllFingers
//
// Purpose:		Deletes all stored finger packets.
//
- (void) popAllFingers
{
	NSUInteger counter = 0;

	// Delete all the local copies we've stored
	for(counter = 0; counter < [self->fingerData count]; counter++)
	{
		NSValue *value = [self->fingerData objectAtIndex:counter];
		FreeFingerCollection( (WacomMTFingerCollection*)[value pointerValue] );
	}

	[self->fingerData removeAllObjects];
}



#pragma mark -
#pragma mark DESTRUCTOR
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
- (void) dealloc
{
	[self popAllFingers];

	[fingerData release];
	[super dealloc];
}



@end


