/*----------------------------------------------------------------------------

FILE NAME
	FingerDataController.m

PURPOSE
	Displays finger data from callbacks from the Wacom MultiTouch API.

COPYRIGHT
	Copyright WACOM Technology, Inc.  2011.
	All rights reserved.

----------------------------------------------------------------------------*/
#import "FingerDataController.h"

#import <WacomMultiTouch/WacomMultiTouch.h>

#import "FingerDataView.h"


@implementation FingerDataController

//////////////////////////////////////////////////////////////////////////////
// awakeFromNib
//
// Purpose:		UI has been loaded.
//
- (void) awakeFromNib
{
	// stash some window geometry for fancy resizing later.
	marginX  = NSWidth([[window contentView] bounds])  - NSWidth([dataView frame]);
	marginY  = NSHeight([[window contentView] bounds]) - NSHeight([dataView frame]);
}



//////////////////////////////////////////////////////////////////////////////
// initWithDeviceID:
//
// Purpose:		Creates a new controller which displays data from a Finger read 
//					callback. 
//
- (id) initWithDeviceID:(int)deviceID displayMode:(eDisplayMode)mode
{
	self = [super init];
	
	[NSBundle loadNibNamed:@"FingerData" owner:self];
	
	WacomMTCapability capabilities      = {};
	CGSize            displaySizeInMM   = CGSizeZero;
	CGSize            touchSizeInMM     = CGSizeZero;
	CGRect            displayRectInPts  = CGRectZero;
	NSSize            contentSize       = NSZeroSize;
	NSSize            windowContentSize = NSZeroSize;
	
	WacomMTGetDeviceCapabilities(deviceID, &capabilities);

	// Display a view which is the exact size of the touch sensor area.
	
	touchSizeInMM     = CGSizeMake(capabilities.PhysicalSizeX, capabilities.PhysicalSizeY);
	displayRectInPts	= CGDisplayBounds(CGMainDisplayID());
	displaySizeInMM   = CGDisplayScreenSize(CGMainDisplayID()); // physical size in mm
	
	contentSize.width    = touchSizeInMM.width  * (displayRectInPts.size.width  / displaySizeInMM.width);
	contentSize.height   = touchSizeInMM.height * (displayRectInPts.size.height / displaySizeInMM.height);
	dataViewAspectRatio	= contentSize.width / contentSize.height;

	// Size the window to physically match the sensor size
	windowContentSize = NSMakeSize(contentSize.width + marginX, contentSize.height + marginY);
	displayMode = mode;
	[dataView setDisplayMode:mode];

	[window setContentSize:windowContentSize];
	[window setTitle:[NSString stringWithFormat:@"Finger Data / Device ID %d", deviceID]];
	[window setFingerController:self];
	// Make the coordinate system of the view use logical units, so the view can 
	// draw finger data directly without units conversion. 
	[dataView setBoundsFromLogicalSize:NSMakeSize(capabilities.LogicalWidth, capabilities.LogicalHeight)];

	return self;
}



#pragma mark -
#pragma mark ACCESSORS
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// window
//
// Purpose:		Returns the viewer window.
//
- (NSWindow *) window
{
	return self->window;
}

-(void) setDisplayMode:(eDisplayMode)mode
{
	displayMode = mode;
	[dataView setDisplayMode:mode];
}

//////////////////////////////////////////////////////////////////////////////
// pushNewFingers:
//
// Purpose:		Adds a new set of finger data to the display.
//
- (void) pushNewFingers:(WacomMTFingerCollection)fingerCollection
{
	[dataView pushNewFingers:fingerCollection];
	
	
	size_t   fingerCounter  = 0;
	size_t   fingerCount    = 0;
	
	for(fingerCounter = 0; fingerCounter < fingerCollection.FingerCount; fingerCounter++)
	{
		// Count the finger ONLY if it is down
		WacomMTFinger *finger = &fingerCollection.Fingers[fingerCounter];
		if(finger->TouchState == WMTFingerStateDown || finger->TouchState == WMTFingerStateHold)
		{
			fingerCount++;
		}
	}
	[self->fingerCountField setIntValue:fingerCount];
}



#pragma mark -
#pragma mark ACTIONS
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// showWindow
//
// Purpose:		
//
- (void) showWindow
{
		[self->window makeKeyAndOrderFront:nil];
}



#pragma mark -
#pragma mark DELEGATES
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// windowWillResize:toSize:
//
// Purpose:		This is just fancy resize code to make sure the window maintains 
//					the aspect ratio of the tablet data view. 
//
- (NSSize)windowWillResize:(NSWindow *)sender toSize:(NSSize)frameSize
{
	NSRect   proposedContentRect  = [sender contentRectForFrameRect:NSMakeRect(0, 0, frameSize.width, frameSize.height)];
	NSSize   newSize              = NSZeroSize;
	
	proposedContentRect.size.width   -= self->marginX;
	proposedContentRect.size.height  -= self->marginY;
	
	proposedContentRect.size.width = NSHeight(proposedContentRect) * dataViewAspectRatio;
	
	proposedContentRect.size.width   += self->marginX;
	proposedContentRect.size.height  += self->marginY;
	
	newSize = [sender frameRectForContentRect:proposedContentRect].size;
	
	return newSize;
}



#pragma mark -
#pragma mark DESTRUCTOR
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
- (void) dealloc
{
	[self->window close];

	[super dealloc];
}



@end
