/*----------------------------------------------------------------------------

FILE NAME
	AppDelegate.mm

PURPOSE
	Application controller for Wacom MultiTouch API sample application.
	
	This sample application connects to the Wacom multitouch API library and 
	registers to receive touch events. 

COPYRIGHT
	Copyright WACOM Technology, Inc.  2011.
	All rights reserved.

----------------------------------------------------------------------------*/
#import "AppDelegate.h"

#import "FingerDataController.h"
#import <WacomMultiTouch/WacomMultiTouch.h>

// Prototypes
void MyAttachCallback(WacomMTCapability deviceInfo, void *userInfo);
void MyDetachCallback(int deviceID, void *userInfo);
int MyFingerCallback(WacomMTFingerCollection *fingerPacket, void *userData);


@implementation AppDelegate

//////////////////////////////////////////////////////////////////////////////
// init
//
// Purpose:		Initialize this controller object.
//
- (id) init
{
	self = [super init];
	
	capabilitiesDataSource  = [[NSMutableArray alloc] init];
	fingerDataViewers       = [[NSMutableDictionary alloc] init];
	displayMode             = eDM_Panel;
	return self;
}



#pragma mark -
#pragma mark ACTIONS
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// initializeTouchAPI:
//
// Purpose:		Register this application with the Wacom touch API.
//
- (IBAction) initializeTouchAPI:(id)sender
{
	// The WacomMultiTouch framework is weak-linked. That means the application 
	// can load if the framework is not present. However, we must take care not 
	// to call framework functions if the framework wasn't loaded. 
	//
	// You can set WacomMultiTouch.framework to be weak-linked in your own 
	// project by opening the Info window for your target, going to the General 
	// tab. In the Linked Libraries list, change the Type of 
	// WacomMultiTouch.framework to "Weak". 
	
	if(WacomMTInitialize != NULL)
	{
		WacomMTError err = WacomMTInitialize(WACOM_MULTI_TOUCH_API_VERSION);

		if(err == WMTErrorSuccess)
		{
			self->isTouchAPIConnected = YES;
			
			[self displayAttachedDevices];
			[self setConstraints];
			
			// Listen for device connect/disconnect.
			// Note that the attach callback will be called for each connected device 
			// immediately after the callback is registered. 
			WacomMTRegisterAttachCallback(MyAttachCallback, self);
			WacomMTRegisterDetachCallback(MyDetachCallback, self);
		}
	}
	else
	{
		// WacomMultiTouch.framework is not installed. Alas!
		NSBeep();
	}
}



//////////////////////////////////////////////////////////////////////////////
// registerWindowCallback:
//
// Purpose:		Create a callback for finger data. This will receive data from
//					all connected touch tablets. when a touch is made in the window
//
- (IBAction) registerWindowCallback:(id)sender
{
	int   deviceIDs[30]  = {};
	int   deviceCount    = 0;
	int   counter        = 0;

	// Add a viewer for each device's data
	displayMode = eDM_Window;
	deviceCount = WacomMTGetAttachedDeviceIDs(deviceIDs, sizeof(deviceIDs));
	if(deviceCount > 30)
	{
		// With a number as big as 30, this will never actually happen.
		NSLog(@"More tablets connected than would fit in the supplied buffer. Will need to reallocate buffer!");
	}
	else
	{
		for(counter = 0; counter < deviceCount; counter++)
		{
			[self addFingerDataControllerForDeviceID:deviceIDs[counter]];
			FingerDataController * controller = [self->fingerDataViewers objectForKey:[NSNumber numberWithInt:deviceIDs[counter]]];

			WacomMTRegisterFingerReadID(deviceIDs[counter], WMTProcessingModeNone, [controller window], 1);
		}
	}
	isWindowCallbackRegistered = YES;
	[self setConstraints];
}



//////////////////////////////////////////////////////////////////////////////
// registerFingerCallback:
//
// Purpose:		Create a callback for finger data. This will receive data from 
//					all connected touch tablets. 
//
- (IBAction) registerFingerCallback:(id)sender
{
	int   deviceIDs[30]  = {};
	int   deviceCount    = 0;
	int   counter        = 0;
	
	// Add a viewer for each device's data
	deviceCount = WacomMTGetAttachedDeviceIDs(deviceIDs, sizeof(deviceIDs));
	if(deviceCount > 30)
	{
		// With a number as big as 30, this will never actually happen.
		NSLog(@"More tablets connected than would fit in the supplied buffer. Will need to reallocate buffer!");
	}
	else
	{
		for(counter = 0; counter < deviceCount; counter++)
		{
			[self addFingerDataControllerForDeviceID:deviceIDs[counter]];
			WacomMTRegisterFingerReadCallback(deviceIDs[counter], NULL, WMTProcessingModeNone, MyFingerCallback, self);
		}
	}
	
	self->isFingerCallbackRegistered = YES;
	[self setConstraints];
}



//////////////////////////////////////////////////////////////////////////////
// quit:
//
// Purpose:		Un-register the Wacom API. All registered callbacks will go dead.
//
- (IBAction) quitTouchAPI:(id)sender
{
	if(WacomMTQuit != NULL) // check API framework availability
	{
		WacomMTQuit();
		
		self->isTouchAPIConnected        = NO;
		self->isFingerCallbackRegistered = NO;
		self->isWindowCallbackRegistered = NO;
		[self displayAttachedDevices];
		[self->fingerDataViewers removeAllObjects];
		[self setConstraints];
	}
}



#pragma mark -
#pragma mark DELEGATES
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// applicationDidFinishLaunching:
//
// Purpose:		The application is loaded, complete with UI.
//
- (void) applicationDidFinishLaunching:(NSNotification *)notification
{
	[self setConstraints];
}



//////////////////////////////////////////////////////////////////////////////
// applicationWillTerminate:
//
// Purpose:		Application is quitting; clean up API connection. This avoids 
//					forcing the driver to manually discard our connection.
//
- (void)applicationWillTerminate:(NSNotification *)notification
{
	if(self->isWindowCallbackRegistered)
	{
		int   deviceIDs[30]  = {};
		int   deviceCount    = 0;
		int   counter        = 0;
		deviceCount = WacomMTGetAttachedDeviceIDs(deviceIDs, sizeof(deviceIDs));
		for(counter = 0; counter < deviceCount; counter++)
		{
			FingerDataController * controller = [self->fingerDataViewers objectForKey:[NSNumber numberWithInt:deviceIDs[counter]]];
			WacomMTUnRegisterFingerReadID([controller window]);
		}
		self->isWindowCallbackRegistered = NO;
	}
	[self quitTouchAPI:NSApp];
}



#pragma mark -
#pragma mark CALLBACKS
#pragma mark -


//////////////////////////////////////////////////////////////////////////////
// deviceDidAttachWithCapabilities:
//
// Purpose:		Called by the touch API callback
//
- (void) deviceDidAttachWithCapabilities:(WacomMTCapability)deviceInfo
{
	[self displayAttachedDevices];
	
	if(self->isFingerCallbackRegistered)
	{
		[self addFingerDataControllerForDeviceID:deviceInfo.DeviceID];
		WacomMTRegisterFingerReadCallback(deviceInfo.DeviceID, NULL, WMTProcessingModeNone, MyFingerCallback, self);
	}
}



//////////////////////////////////////////////////////////////////////////////
// deviceDidDetach:
//
// Purpose:		Called by the touch API callback.
//
- (void) deviceDidDetach:(int)deviceID
{
	[self displayAttachedDevices];
	
	[self->fingerDataViewers removeObjectForKey:[NSNumber numberWithInt:deviceID]];
}



//////////////////////////////////////////////////////////////////////////////
// didReceiveFingerData:
//
// Purpose:		Finger data has arrived! Route to the correct viewer.
//
- (void) didReceiveFingerData:(WacomMTFingerCollection)fingerPacket
{
	int                  deviceID = fingerPacket.DeviceID;
	FingerDataController *viewer  = [self->fingerDataViewers objectForKey:[NSNumber numberWithInt:deviceID]];
	
	[viewer pushNewFingers:fingerPacket];
}



#pragma mark -
#pragma mark TABLE DATA SOURCE
#pragma mark -
// to display the tablet capabilities table

//////////////////////////////////////////////////////////////////////////////
// numberOfRowsInTableView:
//
// Purpose:		Return the number of objects in the capabilities table (the 
//					number of connected tablets) 
//
- (NSInteger) numberOfRowsInTableView:(NSTableView *)aTableView
{
	return [capabilitiesDataSource count];
}



//////////////////////////////////////////////////////////////////////////////
// tableView:objectValueForTableColumn:row:
//
// Purpose:		Provide the data for the tablet capabilities table
//
- (id)            tableView:(NSTableView *)tableView
  objectValueForTableColumn:(NSTableColumn *)tableColumn
								row:(NSInteger)rowIndex
{
	NSString       *identifier    = [tableColumn identifier];
	NSDictionary   *deviceRecord  = nil;
	id             objectValue    = nil;
	
	deviceRecord   = [capabilitiesDataSource objectAtIndex:rowIndex];
	objectValue    = [deviceRecord objectForKey:identifier];
	
	return objectValue;
}



#pragma mark -
#pragma mark UTILITIES
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// addFingerDataControllerForDeviceID:
//
// Purpose:		Creates a new controller object to display finger data for the 
//					given device ID. 
//
- (void) addFingerDataControllerForDeviceID:(int)deviceID
{
	FingerDataController *viewer     = [[FingerDataController alloc] initWithDeviceID:deviceID displayMode:displayMode];
	NSPoint              windowPoint = NSZeroPoint;
	
	// Cascade the windows
	if([self->fingerDataViewers count] == 0)
	{
		NSRect screenRect = [[NSScreen mainScreen] visibleFrame];
		windowPoint = NSMakePoint(NSMinX(screenRect) + 20, NSMaxY(screenRect) + 20);
	}
	else
	{
		NSRect windowFrame = [[[NSApp windows] lastObject] frame];
		windowPoint = NSMakePoint(NSMinX(windowFrame), NSMaxY(windowFrame));
	}

	
	[self->fingerDataViewers setObject:viewer forKey:[NSNumber numberWithInt:deviceID]];
	[[viewer window] cascadeTopLeftFromPoint:windowPoint];
	[viewer showWindow];
	
	[viewer release];
}



//////////////////////////////////////////////////////////////////////////////
// displayAttachedDevices
//
// Purpose:		Gets a list of connected devices and displays them in the device 
//					table. 
//
- (void) displayAttachedDevices
{
	int   deviceIDs[30]  = {};
	int   deviceCount    = 0;
	int   counter        = 0;
	
	// Ask the Wacom API for all connected touch API-capable devices.
	// Pass a comfortably large buffer so you don't have to call this method 
	// twice. 
	deviceCount = WacomMTGetAttachedDeviceIDs(deviceIDs, sizeof(deviceIDs));
	
	if(deviceCount > 30)
	{
		// With a number as big as 30, this will never actually happen.
		NSLog(@"More tablets connected than would fit in the supplied buffer. Will need to reallocate buffer!");
	}
	else
	{
		// Display a list of devices in the table on the application's main window.
	
		[capabilitiesDataSource removeAllObjects];
		
		// Repopulate with current devices
		for(counter = 0; counter < deviceCount; counter++)
		{
			int                  deviceID       = deviceIDs[counter];
			WacomMTCapability    capabilities   = {};
			NSMutableDictionary  *deviceRecord  = [[NSMutableDictionary alloc] init];
			
			WacomMTGetDeviceCapabilities(deviceID, &capabilities);
			
			[deviceRecord setObject:[NSNumber numberWithInt:deviceID] forKey:@"deviceID"];
			[deviceRecord setObject:[NSNumber numberWithInt:capabilities.FingerMax] forKey:@"fingerCount"];
			[deviceRecord setObject:[NSString stringWithFormat:@"%d x %d", capabilities.ReportedSizeX, capabilities.ReportedSizeY] forKey:@"scanSize"];
			
			switch(capabilities.Type)
			{
				case WMTDeviceTypeIntegrated:
					[deviceRecord setObject:@"Integrated" forKey:@"type"];
				break;
				
				case WMTDeviceTypeOpaque:
					[deviceRecord setObject:@"Opaque" forKey:@"type"];
				break;
			}
			
			[capabilitiesDataSource addObject:deviceRecord];
			[deviceRecord release];
		}
	}
	
	[capabilitiesTable reloadData];
}



//////////////////////////////////////////////////////////////////////////////
// setConstraints
//
// Purpose:		Update the enabled/disabled state of UI widgets.
//
- (void) setConstraints
{
	[self->initButton             setEnabled:(isTouchAPIConnected == NO)];
	[self->quitButton             setEnabled:(isTouchAPIConnected == YES)];
	[self->fingerCallbackButton	setEnabled:(isTouchAPIConnected == YES && isFingerCallbackRegistered == NO)];
	[self->RegisterWindowCallBacks	setEnabled:(isTouchAPIConnected == YES && isWindowCallbackRegistered == NO)];

}



@end



#pragma mark -
#pragma mark WACOM TOUCH API C-FUNCTION CALLBACKS
#pragma mark -

//////////////////////////////////////////////////////////////////////////////
// MyAttachCallback()
//
// Purpose:		A new device was connected.
//
void MyAttachCallback(WacomMTCapability deviceInfo, void *userInfo)
{
	AppDelegate *controller = (AppDelegate *)userInfo;
	[controller deviceDidAttachWithCapabilities:deviceInfo];
}



//////////////////////////////////////////////////////////////////////////////
// MyDetachCallback()
//
// Purpose:		A device was unplugged.
//
void MyDetachCallback(int deviceID, void *userInfo)
{
	AppDelegate *controller = (AppDelegate *)userInfo;
	[controller deviceDidDetach:deviceID];
}



//////////////////////////////////////////////////////////////////////////////
// MyFingerCallback()
//
// Purpose:		Fingers are moving on one of the connected devices.
//
int MyFingerCallback(WacomMTFingerCollection *fingerPacket, void *userInfo)
{
	AppDelegate *controller = (AppDelegate *)userInfo;
	[controller didReceiveFingerData:*fingerPacket];
	return 0;
}


